/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  TalerCorebankApiClient,
  TalerMerchantApi,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  SimpleTestEnvironmentNg3,
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  HarnessExchangeBankAccount,
  MerchantService,
  getTestHarnessPaytoForLabel,
  setupDb,
} from "../harness/harness.js";

/**
 * Run a test case with a simple TESTKUDOS Taler environment, consisting
 * of one exchange, one bank and one merchant.
 */
export async function createMyTestkudosEnvironment(
  t: GlobalTestState,
): Promise<SimpleTestEnvironmentNg3> {
  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw-password";
  let exchangePaytoUri = getTestHarnessPaytoForLabel(exchangeBankUsername);

  const exchangeBankAccount: HarnessExchangeBankAccount = {
    wireGatewayAuth: {
      password: exchangeBankPassword,
      username: exchangeBankUsername,
    },
    accountPaytoUri: exchangePaytoUri,
    wireGatewayApiBaseUrl: new URL(
      "accounts/exchange/taler-wire-gateway/",
      bank.baseUrl,
    ).href,
  };

  await exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  const coinCommon = {
    cipher: "RSA" as const,
    durationLegal: "3 years",
    durationSpend: "2 years",
    durationWithdraw: "7 days",
    rsaKeySize: 1024,
    feeDeposit: "TESTKUDOS:0.0025",
    feeWithdraw: "TESTKUDOS:0",
    feeRefresh: "TESTKUDOS:0",
    feeRefund: "TESTKUDOS:0",
  };

  exchange.addCoinConfigList([
    {
      ...coinCommon,
      name: "c1",
      value: "TESTKUDOS:1.28",
    },
    {
      ...coinCommon,
      name: "c2",
      value: "TESTKUDOS:0.64",
    },
    {
      ...coinCommon,
      name: "c3",
      value: "TESTKUDOS:0.32",
    },
    {
      ...coinCommon,
      name: "c4",
      value: "TESTKUDOS:0.16",
    },
    {
      ...coinCommon,
      name: "c5",
      value: "TESTKUDOS:0.08",
    },
    {
      ...coinCommon,
      name: "c5",
      value: "TESTKUDOS:0.04",
    },
    {
      ...coinCommon,
      name: "c6",
      value: "TESTKUDOS:0.02",
    },
    {
      ...coinCommon,
      name: "c7",
      value: "TESTKUDOS:0.01",
    },
  ]);

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  const {accessToken: adminAccessToken} = await merchant.addDefaultInstance();
  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [getTestHarnessPaytoForLabel("minst1")],
  }, {
    adminAccessToken
  });

  console.log("setup done!");

  const { walletClient, walletService } = await createWalletDaemonWithClient(
    t,
    {
      name: "w1",
    },
  );

  return {
    commonDb: db,
    exchange,
    merchant,
    merchantAdminAccessToken: adminAccessToken,
    walletClient,
    walletService,
    bankClient,
    bank,
    exchangeBankAccount,
  };
}

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runFeeRegressionTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchantAdminAccessToken, merchant } =
    await createMyTestkudosEnvironment(t);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:1.92",
  });

  await wres.withdrawalFinishedCond;

  const coins = await walletClient.call(WalletApiOperation.DumpCoins, {});

  // Make sure we really withdraw one 0.64 and one 1.28 coin.
  t.assertTrue(coins.coins.length === 2);

  const order: TalerMerchantApi.Order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:1.30",
    fulfillment_url: "taler://fulfillment-success/thx",
  };

  await makeTestPaymentV2(t, { walletClient, merchant, order, merchantAdminAccessToken });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const txs = await walletClient.call(WalletApiOperation.GetTransactions, {});
  t.assertAmountEquals(txs.transactions[1].amountEffective, "TESTKUDOS:1.30");
  console.log(txs);
}

runFeeRegressionTest.suites = ["wallet"];
