/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  NotificationType,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runAgeRestrictionsPeerTest(t: GlobalTestState) {
  // Set up test environment

  const { bankClient, exchange } = await createSimpleTestkudosEnvironmentV3(
    t,
    defaultCoinConfig.map((x) => x("TESTKUDOS")),
    {
      ageMaskSpec: "8:10:12:14:16:18:21",
    },
  );

  const w1 = await createWalletDaemonWithClient(t, {
    name: "w1",
    persistent: true,
  });
  const w2 = await createWalletDaemonWithClient(t, {
    name: "w2",
    persistent: true,
  });

  const wallet1 = w1.walletClient;
  const wallet2 = w2.walletClient;

  {
    const withdrawalRes = await withdrawViaBankV3(t, {
      walletClient: wallet1,
      bankClient,
      exchange,
      amount: "TESTKUDOS:20",
      restrictAge: 13,
    });

    await withdrawalRes.withdrawalFinishedCond;

    const purse_expiration = AbsoluteTime.toProtocolTimestamp(
      AbsoluteTime.addDuration(
        AbsoluteTime.now(),
        Duration.fromSpec({ days: 2 }),
      ),
    );

    const initResp = await wallet1.client.call(
      WalletApiOperation.InitiatePeerPushDebit,
      {
        partialContractTerms: {
          summary: "Hello, World",
          amount: "TESTKUDOS:1" as AmountString,
          purse_expiration,
        },
      },
    );

    const peerPushReadyCond = wallet1.waitForNotificationCond(
      (x) =>
        x.type === NotificationType.TransactionStateTransition &&
        x.newTxState.major === TransactionMajorState.Pending &&
        x.newTxState.minor === TransactionMinorState.Ready &&
        x.transactionId === initResp.transactionId,
    );

    await peerPushReadyCond;

    const txDetails = await wallet1.call(
      WalletApiOperation.GetTransactionById,
      {
        transactionId: initResp.transactionId,
      },
    );
    t.assertDeepEqual(txDetails.type, TransactionType.PeerPushDebit);
    t.assertTrue(!!txDetails.talerUri);

    const checkResp = await wallet2.call(
      WalletApiOperation.PreparePeerPushCredit,
      {
        talerUri: txDetails.talerUri,
      },
    );

    await wallet2.call(WalletApiOperation.ConfirmPeerPushCredit, {
      transactionId: checkResp.transactionId,
    });

    const peerPullCreditDoneCond = wallet2.waitForNotificationCond(
      (x) =>
        x.type === NotificationType.TransactionStateTransition &&
        x.newTxState.major === TransactionMajorState.Done &&
        x.transactionId === checkResp.transactionId,
    );

    await peerPullCreditDoneCond;
  }
}

runAgeRestrictionsPeerTest.suites = ["wallet"];
runAgeRestrictionsPeerTest.experimental = true;
